
--- Group communication manager
local manager = {}

local messages = require("lac.clientlib.messages")
local utils = require("lac.utils")

local Manager = {}

local function send_grp_msg(self, gm, msgtype)
   local msg = messages.new()
   msg.type = msgtype
   msg:set_content_obj(gm)
   msg.sender_id = self.node.uuid
   msg.add_tag("_groupAPI")
   self.node:send_message(msg)
end

local function ensure_grp_array(grps)
   if grps.is_grp then
      grps = { grps }
   end
   return grps
end

function Manager:send_grpcast_msg(msg, grps, msgtype)
   grps = ensure_grp_array(grps)
   local gcm = messages.new_protobuf(messages.types.GROUPCAST)
   gcm.to_send = grps
   gcm.contents = msg
   send_grp_msg(self, gcm, msgtype or messages.types.GROUPCAST)
end

function Manager:get_joined_grps()
   local copy = {}
   for i, v in ipairs(self.explicit_grps) do
      copy[i] = v
   end
   return copy
end

function Manager:get_all_grps()
   local copy = {}
   for i, v in ipairs(self.implicit_grps) do
      table.insert(copy, v)
   end
   for i, v in ipairs(self.explicit_grps) do
      table.insert(copy, v)
   end
   return copy
end

function Manager:join_grp(grps)
   grps = ensure_grp_array(grps)
   for i, grp in ipairs(self.grps) do
      table.insert(self.explicit_grps, grp)
   end
   send_grp_msg(self, messages.new_membership(grps), messages.types.GROUPMEMBERSHIP)
end

local function send_explicit_grps_sub(self)
   send_grp_msg(self, messages.new_membership(self.explicit_grps), messages.types.GROUPMEMBERSHIP)
end

function Manager:leave_grp(grps)
   grps = ensure_grp_array(grps)
   local set = {}
   for i, v in ipairs(grps) do
      set[v] = true
   end
   for i, grp, remove in utils.ipairs_remove(self.explicit_grps) do
      if set[grp] then
         remove()
      end
   end
   send_grp_msg(self, messages.new_membership(nil, grps), messages.types.GROUPMEMBERSHIP)
end

function Manager:add_listener(kind, listener, arg)
   utils.add_to_listeners_tbl(self.grp_membership_listeners, listener, arg)
end

function Manager:remove_listener(kind, listener)
   utils.remove_from_listeners_tbl(self.grp_membership_listeners, listener)
end

local grp_msg_listener = {

   new_protocol_message_received = function(mgr, node, msgtype, msg)
      if msgtype == messages.types.GROUPMEMBERSHIP then
         local status_data = msg.get_content_obj()
         if #status_data.left > 0 then
            utils.inform_listeners(mgr.grp_membership_listeners, mgr, "leaving_groups", status_data.left)
         end
         if #status_data.joined > 0 then
            utils.inform_listeners(mgr.grp_membership_listeners, mgr, "entering_groups", status_data.joined)
         end
      elseif msgtype == messages.types.GROUPCAST then
         -- groupcast message
         -- nothing here (see GroupCommunicationManager.java:250)
      end
   end,
   
   connected = function(mgr)
      mgr.implicit_grps = {}
   end,
   
   reconnected = function(mgr, node, did_handover, did_mandatory_handover)
      mgr.implicit_grps = {}
      if did_handover then
         mgr:send_explicit_grps_sub()
      end
   end,
   
   disconnected = function(mgr, node)
      mgr.implicit_grps = {}
   end,

}

local Manager_mt = {
   __gc = function(self)
      self.node:remove_advanced_listener(self.grp_msg_listener, messages.types.GROUPCAST)
      self.node:remove_advanced_listener(self.grp_msg_listener, messages.types.GROUPMEMBERSHIP)
   end
}

function manager.new(node)
   local self = {
      node = node,
      grp_msg_listener = grp_msg_listener,
      grp_membership_listeners = {},
      implicit_grps = {},
      explicit_grps = {},
      
      send_grpcast_msg = Manager.send_grpcast_msg,
      get_joined_grps = Manager.get_joined_grps,
      get_all_grps = Manager.get_all_grps,
      join_grp = Manager.join_grp,
      leave_grp = Manager.leave_grp,
      add_listener = Manager.add_listener,
      remove_listener = Manager.remove_listener,
   }
   node:add_advanced_listener(messages.types.GROUPCAST, self.grp_msg_listener, self)
   node:add_advanced_listener(messages.types.GROUPMEMBERSHIP, self.grp_msg_listener, self)
   node:add_listener("external", self.grp_msg_listener, self)
   setmetatable(self, Manager_mt)
   return self
end

return manager

