
local messages = {}

if not unpack then unpack = table.unpack end
local pb = require("pb")
local proto = pb.require("lac.clientlib.protocol.protocol")
local javaserialize = require("javaserialize")

local Message = {}

function Message:index_of_tag(tag)
   for i, t in ipairs(self.tag_list) do
      if tag == t then
         return i
      end
   end
   return -1
end

function Message:has_tag(tag)
   return self:index_of_tag(tag) ~= -1
end

function Message:add_tag(tag)
   if self:index_of_tag(tag) == -1 then
      table.insert(self.tag_list, tag)
   end
end

function Message:remove_tag(tag)
   local idx = self:index_of_tag(tag)
   if idx ~= -1 then
      table.remove(self.tag_list, idx)
   end
end

function Message:tostring()
   return table.concat({
      "{", self.type or "?",
      " gw_logical_time:", self.gw_logical_time or "?",
      " content:", #self.content, "}",
   })
end

function Message:get_content_obj()
   return self.content_obj
end

function Message:set_content_lua_obj(obj, data)
   self.content_obj = obj
   self.content = data or obj
end

function Message:set_content_java_obj(obj, klass)
   if type(obj) == "string" then
      self.content = javaserialize.serialize(obj)
   else
      self.content = javaserialize.serialize(obj, klass)
   end
   self.content_obj = obj
end

messages.types = proto.MSGType
messages.content_types = proto.PayloadSerialization

function messages.new()
   local msg = {
      is_app_msg = true,
      serial_version_id = {0, 1},
      content = "", -- serialized object
      content_obj = nil, -- original object
      content_type = messages.content_types.JAVA,
      tag_list = {},
      sender_id = nil,
      sender_gw_id = nil,
      recipient_id = nil,
      recipient_gw_id = nil,
      gw_logical_time = 0,
      type = messages.types.APPLICATION,
      
      add_tag = Message.add_tag,
      has_tag = Message.has_tag,
      remove_tag = Message.remove_tag,
      index_of_tag = Message.index_of_tag,
      tostring = Message.tostring,
      set_content_lua_obj = Message.set_content_lua_obj,
      set_content_java_obj = Message.set_content_java_obj,
      get_content_obj = Message.get_content_obj,
   }
   return msg   
end

function messages.new_protobuf(msgtype)
   local msg = messages.new()
   msg.type = msgtype
   msg.content_type = messages.content_types.PROTOCOLBUFFER
   return msg
end

function messages.new_service()
   local msg = messages.new()
   msg.is_service = true
   return msg
end

messages.pubsub_mt = {
   __eq = function(a, b)
      return (a.origin_uuid == b.origin_uuid) and (a.properties["creationTime"] == b.properties["creationTime"])
   end
}

function messages.new_membership(joined, left)
   local msg = messages.new_protobuf(messages.types.GROUPMEMBERSHIP)
   msg.joined = joined
   msg.left = left
   return msg
end

function messages.new_pubsub(origin_uuid, info_class, main_value)
   local msg = messages.new_protobuf(messages.types.PUBSUBEVENT)
   msg.info_class = info_class
   msg.origin_uuid = origin_uuid
   msg.info = {}
   msg.properties = {
      ["creationTime"] = tostring(os.time() * 1000),
      ["main"] = main_value,
   }
   setmetatable(msg, messages.pubsub_mt)
   return msg
end

return messages

