
local node = {}

local config = require("lac.clientlib.mrudp.config")
local reader = require("lac.clientlib.mrudp.reader")
local messages = require("lac.clientlib.messages")
local serialization = require("lac.clientlib.serialization")
local client = require("lac.mrudp.client")
local timer = require("lac.timer")
local utils = require("lac.utils")
local uuid = require("uuid")

local Node = {}

local function log(self, level, msg)
   self.logger[level](self.logger, "["..self.uuid.."] "..msg)
end

function Node.debug(self, msg) log(self, "debug", msg) end
function Node.info(self, msg) log(self, "info", msg) end
function Node.warn(self, msg) log(self, "warn", msg) end

function Node.fail(self, msg)
   self.logger:warn(msg)
   return nil, msg
end

local function send_object(self, obj, name)
   if not name then
      name = "lac.cnclib.sddl.message.ClientLibProtocol.ClientLibMessage" -- FIXME
   end
   if self.in_handover or not self.skt.connected then
      return self:fail("Can't send message now. Socket is closed or in handover.")
   end
   self:info("send message - preparing")
   
   local appmsg = obj
   if not appmsg.sender_id then
      appmsg.sender_id = self.sender_uuid
   end
   
   local data, err = serialization.to_protocol_msg(appmsg)
   local header = tostring(#name) .. "|" .. tostring(#data) .. "|"
   
   self:info("send message - writing "..header)
   self.skt:send(header..name..data)
   
   self:info("send message - flush")
   -- self.skt:flush() por algum motivo esse flush ta demorando
   local seqn = self.skt:last_seqn()
   self:info("send message - flushed")
   
   if self.in_handover then
      self:warn("in handover")
   end
   
   if not appmsg.is_service then
      table.insert(self.sent_objects, { sent_object = obj, seqn = seqn })
      self.scheduler:enqueue(self.sent_objects)
   end
   
   return true
end

function Node.set_uuid(self, uuid)
   uuid = utils.binary_uuid(uuid)
   self.uuid = uuid
   self.skt:set_uuid(uuid)
end

---
-- Tries to establish a connection to the specified address.
function Node.connect(self, host, port)
   self.gw_host = host
   self.gw_port = port
   self.skt:add_listener(self.my_listener, self)
   self.skt:add_listener(self.pasv_reader, self.pasv_reader)
   self.skt:set_uuid(self.uuid)
   return self.skt:connect(host, port, 0) -- TODO Java comment: "check timeout, default is 0"
end

function Node.inform_listeners(self, kind, event, ...)
   utils.inform_listeners(self[kind.."_listeners"], self, event, ...)
end

function Node.inform_advanced_listeners(self, event, msgtype, ...)
   local listeners = self.advanced_listeners[msgtype]
   if listeners then
      utils.inform_listeners(listeners, self, event, ...)
   end
end

function Node.add_listener(self, kind, listener, arg)
   utils.add_to_listeners_tbl(self[kind.."_listeners"], listener, arg)
end

function Node.remove_listener(self, kind, listener)
   utils.remove_from_listeners_tbl(self[kind.."_listeners"], listener)
end

function Node.add_advanced_listener(self, msgtype, listener, arg)
   local listeners = self.advanced_listeners[msgtype]
   if not listeners then
      listeners = {}
      self.advanced_listeners[msgtype] = listeners
   end
   utils.add_to_listeners_tbl(listeners, listener, arg)
end

function Node.remove_advanced_listener(self, msgtype, listener)
   local listeners = self.advanced_listeners[msgtype]
   if not listeners then
      return
   end
   utils.remove_from_listeners_tbl(listeners, listener)
end

---
-- Task that performs handover
local function handover_task_fn(timer, self, did_handover)
   if self.handover_step == 0 then
      -- disconnect
      self.handover_step = 1
      self:disconnect()

   -- FIXME: how to go from step 0 to step 1?
   -- Need to understand the logic in the Java version better.
   -- For testing, I just changed the 'elseif' here
   -- to force a fallthrough. Handover seems to run.
   --             -- Hisham, 2014-03-21
   --
   -- elseif self.handover_step == 1 then
   end
   if self.handover_step == 1 then

      -- connect
      local addr = self.gw_host..":"..self.gw_port
      self:info("handover connect at "..addr)
      self.skt:remove_listener(self.my_listener)
      self.skt:remove_listener(self.pasv_reader)
      self.skt = client.new(self.scheduler, nil, nil, nil, self.config)
      self.handover_step = 2
      self:info("handover connecting at "..addr)
      local ok = self:connect(self.gw_host, self.gw_port)
      if ok then
         self:info("handover waiting connection - "..self.uuid.." at "..addr)
      else
         self:info("handover can't connect - at "..addr)
      end
   end
end

---
-- Updates the gateway IP address.
-- @param must_switch boolean indicating if must choose another one
-- @param mandatory boolean indicating if handover is mandatory
-- @return true if the gateway IP was updated, false otherwise.
local function update_poa(self, must_switch, mandatory)
   self:info("updating PoA?")
   if not self.poa or #self.poa.gw_list == 0 then
      self:info("inexistent PoA list")
      return false
   end
   local gw_list = self.poa.gw_list
   
   local poa_index = mandatory and 1 or math.random(#gw_list)
   local new_addr = gw_list[poa_index]
   
   local addr = self.gw_host..":"..self.gw_port
   -- Same GW and port
   if addr == new_addr then -- FIXME check if format sent via POA matches addr
      if not must_switch or (mandatory and #gw_list <= 1) then
         return false
      else
         local another_idx = math.random(#gw_list - 1)
         if another_idx >= poa_index then
            another_idx = another_idx + 1
         end
         new_addr = gw_list[another_idx]
         self:debug("get diff PoA ok")
      end
   end
   
   self.gw_host, self.gw_port = new_addr:match("([^:]+):(.*)")
   self.gw_port = tonumber(self.gw_port)
   return true
end

---
-- Starts the handover process.
-- @param must_switch if must choose another one
-- @param mandatory if it was mandatory
function Node.start_handover(self, must_switch, mandatory)
   local did_handover = update_poa(self, must_switch, mandatory)
   
   if did_handover then
      self.in_handover = true
      self:inform_listeners("sddl", "starting_handover", self.gw_host, self.gw_port, mandatory)
   end
   
   self.did_handover = did_handover
   self.did_mandatory_handover = mandatory
   
   if did_handover or not self.skt.connected then
      self:info("create handover task")

      if self.skt.connected then
         self.handover_step = 0
      else
         self.handover_step = 1
      end
      self.handover_task = timer.new(self.scheduler, "handover_task", handover_task_fn, self, did_handover)
      self.handover_task:schedule(1 + math.random() * 7)
   else
      self:debug("no need to handover")
   end
end

---
-- Disconnect from the gateway.
function Node.disconnect(self)
   self:warn("about to be disconnected")
   if not self.skt.connected then
      return self:fail("Socket not connected")
   end
   self.skt:close()
   if not self.in_handover then
      self.skt:remove_listener(self.my_listener)
      self.skt:remove_listener(self.pasv_reader)
      self.scheduler:enqueue(self.objects_to_send, "die")
   end
end

---
-- Sends a message to the Gateway.
function Node.send_message(self, msg)
   self:debug("send message")
   table.insert(self.objects_to_send, msg)
   self.scheduler:enqueue(self.objects_to_send, true)
   self:debug("message queued")
end

local function sender_fn(self)
   while self.skt.connected and not self.in_handover do
      if #self.objects_to_send == 0 then
         local ok, cmd = self.scheduler:wait(self.objects_to_send)
         if cmd == "die" then break end
      end
      local obj = table.remove(self.objects_to_send, 1)
      if obj then
         local ok, err = send_object(self, obj)
         if not ok then
            table.insert(self.objects_to_send, 1, obj)
         end
      end
   end
   self.sender_coro = nil
   self.scheduler:notify(self.my_listener.connection_opened)
end

local function notify_unsent_messages(self)
   local unsent = {}
   for i, msg, remove in utils.ipairs_remove(self.sent_objects) do
      if msg.sent_object.is_app_msg then
         table.insert(unsent, msg.sent_object)
         remove()
      end
   end
   for i, msg, remove in utils.ipairs_remove(self.objects_to_send) do
      if msg.sent_object.is_app_msg then
         table.insert(unsent, msg.sent_object)
         remove()
      end
   end
   self:inform_listeners("external", "unsent_messages", unsent)
end

local function reset_sent_to_unsent(self)
   local sent_messages = {}
   for i, msg, remove in utils.ipairs_remove(self.sent_objects) do
      if msg.sent_object.is_app_msg then
         table.insert(sent_messages, msg.sent_object)
         remove()
      end
   end
   while #sent_messages > 0 do
      local msg = table.remove(sent_messages)
      table.insert(self.objects_to_send, 1, msg)
   end
   self.scheduler:enqueue(self.objects_to_send, true)
end

local my_listener = {

   connection_opened = function(self, clientskt)
      self:debug("connection opened")
      
      self.pasv_reader:reset()
      self.n_connections = self.n_connections + 1
      self.reconnection_attempts = 0
      self.reconnecting = false
      self.in_handover = false
      self.handover_task = nil
      self:warn("closing the current alive sender thread")

      if self.sender_coro then
         -- terminate sender_fn
         self.scheduler:enqueue(self.objects_to_send, "die")
         self.scheduler:wait(self.my_listener.connection_opened)
      end
      assert(not self.sender_coro)

      self.sender_coro = coroutine.create(sender_fn)
      local ok, err = coroutine.resume(self.sender_coro, self)
      if not ok then error(err, 2) end
      
      self:inform_listeners("external", self.n_connections == 1 and "connected" or "reconnected", self.did_handover, self.did_mandatory_handover)
      self.did_handover = false
      self.did_mandatory_handover = false
   end,
   
   connection_refused = function(self, clientskt)
      self:debug("connection refused")

      if self.in_handover or self.reconnecting and self.reconnection_attempts < self.max_reconnections then
         self.reconnection_attempts = self.reconnection_attempts + 1
         self.reconnecting = true
         
         self:info("connection refused - start new handover")
         self:start_handover(true, false)
      else
         self:info("connection refused - give up")
         notify_unsent_messages(self)
         self:inform_listeners("external", "disconnected")
      end
   end,

   connection_closed = function(self, clientskt)
      self:info("connection closed to "..self.gw_host..":"..self.gw_port)
      
      reset_sent_to_unsent(self)
      if self.in_handover then
         self:info("handover connect again")
         self.handover_task:schedule(1)
         return
      end
      self:info("closed connect again")
      self.did_handover = false
      self.did_mandatory_handover = false
      
      if self.gw_host == nil or self.reconnection_attempts >= self.max_reconnections then
         notify_unsent_messages(self)
         self:inform_listeners("external", "disconnected")
      else
         self.reconnection_attempts = self.reconnection_attempts + 1
         self.reconnecting = true
         
         self:info("closed, start handover")
         self:start_handover(false, false)
      end
   end,

   connection_failure = function(self, clientskt)
      -- Note by Lincoln in original Java code:
      -- "since the connection never returns, use as disconnect" -- MrUdpNodeConnection:624
   end, 

   connection_reset = nil,
}

function node.new(scheduler, skt, sender_uuid, cfg, logger)
   cfg = cfg or config.new()
   if not logger then
      require("logging.syslog")
      logger = logging.syslog("lac.clientlib")
   end
   local self = {
      scheduler = scheduler,
      logger = logger,
      uuid = uuid.new(),
      sender_uuid = sender_uuid or uuid.new(),
      gw_host = nil,
      gw_port = nil,
      skt = skt or client.new(scheduler, nil, nil, nil, cfg),
      config = cfg,
      external_listeners = {},
      advanced_listeners = {},
      sddl_listeners = {},
      n_connections = 0,
      max_reconnections = 5,
      objects_to_send = {},
      sent_objects = {},
      in_handover = false,
      did_handover = false,
      did_mandatory_handover = false,
      n_nodeconnections = 0,
      poa = nil,
      
      debug = Node.debug,
      info = Node.info,
      warn = Node.warn,
      fail = Node.fail,
      set_uuid = Node.set_uuid,
      connect = Node.connect,
      start_handover = Node.start_handover,
      disconnect = Node.disconnect,
      send_message = Node.send_message,
      inform_listeners = Node.inform_listeners,
      inform_advanced_listeners = Node.inform_advanced_listeners,
      add_listener = Node.add_listener,
      remove_listener = Node.remove_listener,
      add_advanced_listener = Node.add_advanced_listener,
      remove_advanced_listener = Node.remove_advanced_listener,
   }
   
   self.pasv_reader = reader.get_listener(self)
   self.my_listener = my_listener
   
   if self.skt.connected then
      self.skt:add_listener(self.my_listener, self)
      self.skt:add_listener(self.pasv_reader, self.pasv_reader)
      self.sender_coro = coroutine.create(sender_fn)
      local ok, err = coroutine.resume(self.sender_coro, self)
      if not ok then error(err, 2) end
   end
   return self
end

return node
