
local manager = {}

local Manager = {}

local PUBSUB_CONTROL_GROUP = 100
local PUBSUB_DATA_GROUP = 101

local messages = require("lac.clientlib.messages")
local utils = require("lac.utils")

-- This is not the same algorithm as Java hashCode,
-- but it's superficially similar.
local function hash(s)
   local code = 0
   local flip = 1
   for i = 1, #s do
      local ch = s:sub(i,i):byte()
      code = (code + ch * ((#s-i+1)*31)) % 2^32
   end
   return code
end

local function new_grp(grptype, info_class)
   return { type = grptype, id = hash(info_class) }
end

function Manager:register_pub(info_class)
   self.gcm:join_grp(new_grp(PUBSUB_CONTROL_GROUP, info_class))
end

function Manager:register_pub(info_class)
   self.gcm:leave_grp(new_grp(PUBSUB_CONTROL_GROUP, info_class))
end

local function send_pubsub_msg(self, info_data, grp, msgtype) 
   local psm = messages.new_pubsub()
   psm:set_content_lua_obj(info_data)
   psm.sender_id = self.node.uuid
   psm:add_tag("_pubsubAPI")
   psm.type = msgtype
   self.gcm:send_grpcast_msg(psm, grp, msgtype)
end

function Manager:publish(info_data)
   info_data.properties["senderUuid"] = utils.text_uuid(self.node.uuid)
   info_data.properties["publicationTime"] = tostring(os.time * 1000)
   local grp = new_grp(PUBSUB_DATA_GROUP, info_data.info_class)
   send_pubsub_msg(self, info_data, grp, messages.types.PUBSUBEVENT)
end

local function process_sub(self, sub_id)
   self.gcm:join_grp(new_grp(PUBSUB_DATA_GROUP, sub_id.sub.info_class))
   send_pubsub_msg(self, sub_id.sub, new_grp(PUBSUB_CONTROL_GROUP, sub_id.sub.info_class), messages.type.SUBSCRIPTION)
end

function Manager:subscribe(sub, listener)
   local sub_id = { sub = sub, listener = listener }
   process_sub(self, sub_id)
   table.insert(self.sub_ids, sub_id)
   return sub_id
end

local function remove(tbl, entry)
   for i, v in ipairs(tbl) do
      if v == entry then
         table.remove(tbl, i)
         return true
      end
   end
   return false
end

function Manager:unsubscribe(sub_id)
   return remove(self.sub_ids, sub_id)
end

function Manager:add_listener(listener) 
   utils.add_to_listeners_tbl(self.pubsub_listeners, listener)
end

function Manager:remove_listener(listener) 
   utils.remove_from_listeners_tbl(self.pubsub_listeners, listener)
end

local Manager_mt = {
   __gc = function(self)
      self.node:remove_advanced_listener(self.advanced_listener, messages.types.SUBSCRIPTION)
      self.node:remove_advanced_listener(self.advanced_listener, messages.types.PUBSUBEVENT)
   end
}

local pubsub_msg_listener = {

   new_protocol_message_received = function(mgr, node, msgtype, msg)
      if msgtype == messages.types.SUBSCRIPTION then
         local sub = msg.content_obj
         utils.inform_listeners(mgr.pubsub_listeners, mgr, "new_information_received", sub)
      elseif msgtype == messages.types.PUBSUBEVENT then
         local info_data = msg.content_obj
         utils.inform_listeners(mgr.pubsub_listeners, mgr, "new_information_received", info_data)
         for _, sub_id in mgr.sub_ids do
            if sub_id.sub.info_class == info_data.info_class then
               utils.inform_listeners({ listener = sub_id.listener }, mgr, "new_information_received", info_data, sub)
            end
         end
      end
   end,

   -- TODO
}

function manager.new(node, gcm)
   local self = {
      node = node,
      gcm = gcm,
      advanced_listener = pubsub_msg_listener,
      sub_ids = {},
      pubsub_listeners = {},
      
      register_pub = Manager.register_pub,
      unregister_pub = Manager.unregister_pub,
      publish = Manager.publish,
      subscribe = Manager.subscribe,
      unsubscribe = Manager.unsubscribe,
      add_listener = Manager.add_listener,
      remove_listener = Manager.remove_listener,
   }
   node:add_advanced_listener(messages.types.SUBSCRIPTION, self.advanced_listener, self)
   node:add_advanced_listener(messages.types.PUBSUBEVENT, self.advanced_listener, self)
   setmetatable(self, Manager_mt)
   return self
end

return manager

