
local serialization = {}

local messages = require("lac.clientlib.messages")
local utils = require("lac.utils")
if not unpack then unpack = table.unpack end

local pb = require("pb")

-- FIXME added this because I didn't want to fix the
-- "import" statements in the .protocol files at this point.
pb.path = pb.path .. ";./lac/clientlib/?.proto;"

local proto = pb.require("lac.clientlib.protocol.protocol")
local group_membership_message = pb.require("lac.clientlib.protocol.group_membership_message")
local groupcast_message = pb.require("lac.clientlib.protocol.groupcast_message")
local ping_message = pb.require("lac.clientlib.protocol.ping_message")
local points_of_attachment_message = pb.require("lac.clientlib.protocol.points_of_attachment_message")
local subscribe_message = pb.require("lac.clientlib.protocol.subscribe_message")
local publish_message = pb.require("lac.clientlib.protocol.publish_message")

local function encode_uuid(uuid)
   if not uuid then return nil end
   local binuuid = utils.binary_uuid(uuid)
   return proto.UUID({
      mostSignBits = binuuid:sub(1,8),
      leastSignBits = binuuid:sub(9,16),
   })
end

local function decode_uuid(encoded)
   if not encoded then return nil end
   local raw = encoded[".raw"]
   local out = {}
   for c in (raw.mostSignBits..raw.leastSignBits):gmatch(".") do
      out[#out+1] = ("%02x"):format(c:byte())
   end
   local parts = { table.concat(out):match("(........)(....)(....)(....)(............)") }
   return table.concat(parts, "-")
end

local uuid_map = {
   senderID = "sender_id",
   senderGatewayUUID = "sender_gw_id",
   recipientID = "recipient_id",
   recipientGatewayUUID = "recipient_gw_id",
}

local function encode_map(tbl)
   if not tbl then return {} end
   local map = { entries = {} }
   for k,v in pairs(tbl) do
      map.entries[#map+1] = { key = k, value = v }
   end
   return map
end

local function decode_map(map)
   if (not map) or not map.entries then return {} end
   local tbl = {}
   for _, entry in ipairs(map.entries) do
      tbl[entry.key] = entry.value
   end
   return tbl
end

function serialization.to_protocol_msg(appmsg)
   local tag_list = { "_default" }
   for i, tag in ipairs(appmsg.tag_list) do
      table.insert(tag_list, tag)
   end
   
   if appmsg.type == messages.types.PUBSUBEVENT then
      appmsg.content = publish_message.PublishInformation({
         information = encode_map(appmsg.information),
         properties = encode_map(appmsg.properties),
         originUuid = encode_uuid(appmsg.origin_uuid),
         informationClass = appmsg.information_class,
      }):Serialize()
      appmsg.content_type = messages.content_types.PROTOCOLBUFFER
   end
   
   local msg = proto.ClientLibMessage({
      msgType = appmsg.type,
      gatewayLogicalTime = appmsg.gw_logical_time,
      contentPayload = appmsg.content,
      contentType = appmsg.content_type,
      tagList = tag_list
   })
   for mname, aname in pairs(uuid_map) do
      msg[mname] = encode_uuid(appmsg[aname])
   end
   return msg:Serialize()
end

local function convert_groups(protogroups)
   local tbl = {}
   for i, group in ipairs(protogroups) do
      table.insert(tbl, { type = group.groupType, id = group.groupID })
   end
   return tbl
end

local function deserialize_group_membership(msg, data)
   local gm = group_membership_message.GroupMembership():Parse(data)
   msg.joined = convert_groups(gm.groupsJoined)
   msg.left = convert_groups(gm.groupsLeft)
end

local function deserialize_groupcast(msg, data)
   local gm = groupcast_message.Groupcast():Parse(data)
   msg.to_send = convert_groups(gm.groupsToSend or {})
   msg.contents = gm.content
end

local function deserialize_ping(msg, data)
   local pm = ping_message.Ping():Parse(data)
   msg.id = pm.id
   msg.pong = pm.pong
end

local function deserialize_poa(msg, data)
   local pm = points_of_attachment_message.PointsOfAttachment():Parse(data)
   msg.switch_gw = pm.switchGateway
   msg.gw_list = pm.gatewayList   
end

local function deserialize_subscription(msg, data)
   --deserialize_groupcast(msg, data)
   local pm = subscribe_message.SubscribeInformation():Parse(data)
   msg.topic = pm.topic
   -- not working with filters right now (see Serialization.java:502)
end

local function deserialize_pubsub(msg, data)
   --deserialize_groupcast(msg, data)

   local pm = publish_message.PublishInformation():Parse(data)
   msg.info_class = pm.informationClass
   msg.info = decode_map(pm.information)
   msg.properties = decode_map(pm.properties)
   msg.origin_uuid = decode_uuid(pm.originUuid)
   -- not working with filters right now (see Serialization.java:502)
end

local function deserialize_payload(msg, data, type)
   if type == messages.types.GROUPMEMBERSHIP then
      deserialize_group_membership(msg, data)
   elseif type == messages.types.GROUPCAST then
      deserialize_groupcast(msg, data)
   elseif type == messages.types.PING then
      deserialize_ping(msg, data)
   elseif type == messages.types.POA then
      deserialize_poa(msg, data)
   elseif type == messages.types.SUBSCRIPTION then
      deserialize_subscription(msg, data)
   elseif type == messages.types.PUBSUBEVENT then
      deserialize_pubsub(msg, data)
      setmetatable(msg, messages.pubsub_mt)
   else
      -- app-specific
      msg.content = data
      return
   end
   msg:set_content_lua_obj(msg, data)
end

function serialization.from_protocol_msg(bytes)
   local msg = proto.ClientLibMessage():Parse(bytes)
   local appmsg = messages.new()
   for mname, aname in pairs(uuid_map) do
      appmsg[aname] = decode_uuid(msg[mname])
   end
   appmsg.gw_logical_time = msg.gatewayLogicalTime
   if msg.tagList then
      for i, tag in ipairs(msg.tagList) do
         -- FIXME convert tags?
         table.insert(appmsg.tag_list, tag)
      end
   end
   local msgtype = proto.MSGType[msg.msgType]
   appmsg.type = msgtype
   appmsg.content_type = proto.PayloadSerialization[msg.contentType]
   deserialize_payload(appmsg, msg.contentPayload, msgtype)
   return appmsg
end

return serialization
