
local config = {}

local function check_value(var, min, max)
   return var == nil or (var >= min and var <= max)
end

--- Creates a configuration profile with the default MR-UDP parameter values.
-- If any parameter is nil, it is replaced with the default value.
-- @param max_sendq_size maximum send queue size (packets).
-- @param max_recvq_size maximum receive queue size (packets).
-- @param max_seg_size maximum segment size (octets) (must be at least 22).
-- @param max_ostand maximum number of outstanding segments.
-- @param max_retx maximum number of consecutive retransmissions (0 means unlimited).
-- @param max_cack maximum number of unacknowledged received segments.
-- @param max_outseq maximum number of out-of-sequence received segments.
-- @param max_autorst maximum number of consecutive auto resets (not used).
-- @param nilseg_to null segment timeout (seconds).
-- @param retx_to retransmission timeout (seconds).
-- @param cack_to cumulative acknowledge timeout (seconds).
function config.new(max_sendq_size, max_recvq_size, max_seg_size, max_ostand,
                    max_retx, max_cack, max_outseq, max_autorst,
                    nilseg_to, retx_to, cack_to)
   
   assert(check_value(max_sendq_size, 1, 255), "invalid max send queue size")
   assert(check_value(max_recvq_size, 1, 255), "invalid max recv queue size")
   assert(check_value(max_seg_size, 22, 7990), "invalid max segment size")
   assert(check_value(max_ostand, 1, 255), "invalid max outstanding segments")
   assert(check_value(max_retx, 0, 255), "invalid max retransmissions")
   assert(check_value(max_cack, 0, 255), "invalid max cumulative acks")
   assert(check_value(max_outseq, 0, 255), "invalid max out of sequence")
   assert(check_value(max_autorst, 0, 255), "invalid max auto reset")
   assert(check_value(nilseg_to, 0, 65.535), "invalid null segment timeout")
   assert(check_value(retx_to, 0.1, 65.535), "invalid retransmission timeout")
   assert(check_value(cack_to, 0.1, 65.535), "invalid cumulative ack timeout")

   return {
      max_sendq_size = max_sendq_size or 64,
      max_recvq_size = max_recvq_size or 64,
      max_seg_size = max_seg_size or 256,
      max_ostand = max_ostand or 5,
      -- Note: According to the RUDP protocol's draft, the default maximum
      -- number of retransmissions is 3. However, if packet drops are too
      -- high, the connection may get stall unless the sender continues to
      -- retransmit packets that have not been unacknowledged. We will use
      -- 0 instead, which means unlimited.
      max_retx = max_retx or 10,
      max_cack = max_cack or 3,
      max_outseq = max_outseq or 3,
      max_autorst = max_autorst or 10,
      nilseg_to = nilseg_to or 30,
      retx_to = retx_to or 0.9,
      cack_to = cack_to or 0.6,
   }
end

local default_config = config.new()

function config.get_default_config()
   return default_config
end

return config

