
--- @module lac.mrudp.server
local server = {}

local ServerSkt = {}

local socket = require("socket")
local client = require("lac.mrudp.client")
local segments = require("lac.mrudp.segments")

local DEFAULT_BACKLOG_SIZE = 50

---
-- Waits for a remote connection on the server object and returns a
-- client object representing that connection.
-- @param self MR-UDP object
-- @return If a connection is successfully initiated, a client object
-- is returned. If a timeout condition is met, the method returns nil
-- followed by the error string 'timeout'. Other errors are reported
-- by nil followed by a message describing the error. 
function ServerSkt.accept(self)
   if self.closed then
      return nil, "socket is closed"
   end
   local timeout_time = (self.timeout and self.timeout > 0) and self.scheduler:time() + self.timeout or nil
   
   while #self.backlog == 0 do
      local ok, err = self.scheduler:wait_until(self.backlog, timeout_time)
      if err == "timeout" then
         return nil, err
      end
   end
   return table.remove(self.backlog, 1)
end

--- 
-- Binds the socket to a local address.
-- @param self MR-UDP object
-- @param host string: local IP address; may be "*",
-- meaning all local interface (using INADDR_ANY).
-- @param port number: local port number; may be 0,
-- meaning an ephemeral port.
-- @return true if successful, or nil followed by an error message.
function ServerSkt.bind(self, host, port)
   if self.is_bound then
      return nil, "socket is already bound"
   end
   local ok, err = self.skt:setsockname(host, port)
   if not ok then
      return nil, err
   end
   self.is_bound = true
   return true
end

---
-- Returns the local address information associated to the object.
-- @param self MR-UDP object
-- @return a string with local IP address and a number with the port.
-- In case of error, the method returns nil. 
function ServerSkt.getsockname(self)
   return self.skt:getsockname()
end

local function notify_backlog(self, ...)
   self.scheduler:enqueue(self.backlog, ...)
   self.scheduler:notify(self.connection_opened_lock)
end

function ServerSkt.close(self)
   if self.closed then
      return
   end
   self.closed = true
   while true do
      local clientskt = table.remove(self.backlog)
      if not clientskt then break end
      clientskt:close()
   end
   notify_backlog(self, nil, "socket closed")
   self.scheduler:enqueue(self.skt)
   self.skt:close()
   self.uuids = {}
end

local function remove_client_skt(self, host, port)
   local endpoint = host..":"..port
   local uuid = self.uuid_reverse_lookup[endpoint]
   if uuid then
      self.uuids[uuid] = nil
      self.uuid_reverse_lookup[endpoint] = nil
   end
   local clientskt = self.clients[endpoint]
   self.clients[endpoint] = nil
   return clientskt
end

ServerSkt.listener = {
   
   connection_opened = function(self, clientskt)
      while #self.backlog > self.backlog_size do
         self.scheduler:wait(self.connection_opened_lock)
      end
      table.insert(self.backlog, clientskt)
      notify_backlog(self)
   end,
   
   connection_closed = function(self, clientskt)
      remove_client_skt(self, clientskt.host, clientskt.port)
   end,

   connection_failure = function(self, clientskt)
      remove_client_skt(self, clientskt.host, clientskt.port)
   end,
   
}

local function add_client_skt(self, host, port, endpoint)
   if #(self.backlog) == self.backlog_size then
      return nil
   end
   local clientskt, err = client.new(self.scheduler, host, port, self)
   assert(clientskt, err)
   clientskt:add_listener(ServerSkt.listener, self)
   self.clients[endpoint] = clientskt
   --table.insert(self.backlog, clientskt)
   --self.scheduler:enqueue(self.backlog, clientskt)
   return clientskt
end

local function receiver_fn(self)
   self.scheduler:set_thread_name("server receiver_fn")

   while true do
      local ok, data, host, port = self.scheduler:wait(self.skt)
      if (not ok) or (not data) then
         break
      end
      local endpoint = host..":"..port
      local seg = segments.parse(data)
      local clientskt = self.clients[endpoint]
      if seg.type == "UID" then
         local uuid = seg.uuid
         local registered_endpoint = self.uuids[uuid]
         if registered_endpoint then
            if endpoint ~= registered_endpoint then
               -- update UID segment from different endpoint
               clientskt = self.clients[registered_endpoint]
               
               self.clients[registered_endpoint] = nil
               self.uuid_reverse_lookup[registered_endpoint] = nil
               
               self.uuids[uuid] = endpoint
               self.uuid_reverse_lookup[endpoint] = uuid
               self.clients[endpoint] = clientskt
               
               clientskt.host = host
               clientskt.port = port
            else
               -- ignore UID segment from unchanged endpoint
            end
         else
            self.uuids[uuid] = endpoint
            self.uuid_reverse_lookup[endpoint] = uuid
         end
      elseif seg.type == "SYN" then
         if not clientskt then
            clientskt = add_client_skt(self, host, port, endpoint)
         end
      end
      if clientskt then
         self.scheduler:enqueue(clientskt, seg, host, port)
      else
         -- drop segment
      end
   end
   self.receiver_coro = nil
end

ServerSkt.mt = {
   __gc = function(self)
      self:close()
   end,
}

--- Create a new server socket.
-- @param port number: the port number
-- @param backlog number: the size of the backlog. If 0 or not given, use the
-- default backlog size.
-- @param bindaddr string: the bind address. If not given, use "*".
-- @param scheduler table: a scheduler object. If not given, use the default
-- internal scheduler.
-- @return a socket object.
function server.new(scheduler, port, backlog, bindaddr)
   assert(scheduler, "scheduler not given")
   if not port then port = 0 end
   if not backlog or backlog <= 0 then backlog = DEFAULT_BACKLOG_SIZE end
   if not bindaddr then bindaddr = "*" end -- TODO unused?
   local skt = socket.udp()
   local self = {
      skt = skt,
      backlog_size = backlog,
      backlog = {},
      scheduler = scheduler,
      uuids = {},
      uuid_reverse_lookup = {},
      clients = {},
      
      connection_opened_lock = {},
      
      timeout = -1,
      bound = false,
      closed = false,
      accept = ServerSkt.accept,
      bind = ServerSkt.bind,
      getsockname = ServerSkt.getsockname,
      close = ServerSkt.close,
   }
   setmetatable(self, ServerSkt.mt)
   self:bind(bindaddr, port)
   self.receiver_coro = coroutine.create(receiver_fn)
   local ok, err = coroutine.resume(self.receiver_coro, self)
   if not ok then error(err, 2) end
   return self
end

return server
