
--- @module lac.mrudp.client
local client = {}

local ClientSkt = {}

require("compat52")

local socket = require("socket")
local segments = require("lac.mrudp.segments")
local counter = require("lac.mrudp.counter")
local config = require("lac.mrudp.config")
local tests = require("lac.mrudp.tests")
local timer = require("lac.timer")
local mutex = require("lac.mutex")
local utils = require("lac.utils")

local default_config = config.get_default_config()

local CMP = counter.compare_seqn

-- There is not an active or pending connection
local CLOSED = 0 
-- Request to connect received, waiting ACK
local SYN_RCVD = 1
-- Request to connect sent
local SYN_SENT = 2
-- Data transfer state
local ESTABLISHED = 3
-- Request to close the connection
local CLOSE_WAIT = 4

math.randomseed(os.time())

--- Debug function to use when printing segment queues.
local function print_q(q)
   local t = {}
   for i = 1, #q do
      table.insert(t, q[i].seqn)
   end
   return "[ "..table.concat(t, " ").." ]"
end

---
-- Sets the ACK flag and number of a segment if there is at least one
-- received segment to be acknowledged.
local function check_and_set_ack(self, seg)
   if self.ctr:getreset_cack() == 0 then
      return
   end
   seg:set_ack(self.ctr:get_last())
end

---
-- Sends a segment piggy-backing any pending acknowledgments,
-- writing it out to the underlying UDP socket.
local function send_seg(self, seg)
   if seg.type == "DAT" or seg.type == "RST"
   or seg.type == "FIN" or seg.type == "NUL" then
      check_and_set_ack(self, seg)
   end
   if seg.type == "DAT" or seg.type == "RST"
   or seg.type == "FIN" or seg.type == "UID" then
      assert(self.timers.nilseg:reset())
   end
   tests.log_packet(seg, "send")
   local ret, err = self.skt:sendto(seg:tobytes(), self.host, self.port)
   -- Is udpSleepMillis necessary? Does not seem to be used in Java.
   return ret, err
end

local function inform_listeners(self, event, ...)
   utils.inform_listeners(self.listeners, self, event, ...)
end

function ClientSkt.add_listener(self, listener, arg)
   utils.add_to_listeners_tbl(self.listeners, listener, arg)
end

function ClientSkt.remove_listener(self, listener)
   utils.remove_from_listeners_tbl(self.listeners, listener)
end

---
-- Sends a segment and queues a copy of it in the queue of unacknowledged
-- segments.
local function send_and_q_seg(self, seg)
   self.send_and_q_mutex:lock()
   while (#self.unacked_sent_q >= self.cfg.max_sendq_size
      or self.ctr:get_ostand() > self.cfg.max_ostand)
      and self.connected and not self.closed do
      self.scheduler:wait(self.unacked_sent_q)
   end
   if self.closed then
      self.send_and_q_mutex:unlock()
      return nil, "closed"
   end
   self.ctr:inc_ostand()
   table.insert(self.unacked_sent_q, seg)

   if seg.type ~= "EAK" and seg.type ~= "ACK" then
      if self.timers.retx.idle then
         assert(self.timers.retx:schedule(self.cfg.retx_to, self.cfg.retx_to))
      end
   end
   
   local ok, err = send_seg(self, seg)
   if ok and seg.type == "DAT" then
      inform_listeners(self, "packet_sent")
   end
   self.send_and_q_mutex:unlock()
   return ok, err
end

local function random_seqn()
   return math.random(counter.MAX_SEQUENCE_NUMBER)
end

local function make_syn(self)
   -- synchronize sequence numbers
   local seqn = self.ctr:set_seqn(random_seqn())
   return segments.new("SYN", seqn,
      self.cfg.max_ostand, self.cfg.max_seg_size, self.cfg.retx_to,
      self.cfg.cack_to, self.cfg.nilseg_to,
      self.cfg.max_retx, self.cfg.max_cack, self.cfg.max_outseq,
      self.cfg.max_autorst)
end

local receiver_fn

function ClientSkt.connect(self, host, port, timeout)
   assert(type(host) == "string")
   assert(type(port) == "number")
   if self.closed then
      return nil, "closed"
   end
   if self.connected then
      return nil, "already connected"
   end
   if not self.receiver_coro then
      self.receiver_coro = coroutine.create(receiver_fn)
      local ok, err = coroutine.resume(self.receiver_coro, self)
      if not ok then error(err, 2) end
   end

   if timeout <= 0 then timeout = nil end

   local syn = make_syn(self)

   self.host = host
   self.port = port
   self.timeout = timeout
   
   self.state = SYN_SENT
   local ok, err = send_and_q_seg(self, syn)
   if not ok then
      return nil, err
   end

   local ok, err = self.scheduler:wait(self.connection_lock, self.timeout)
   if self.state == ESTABLISHED then
      return true
   end
   
   self.unacked_sent_q = {}
   
   self.ctr:reset()
   assert(self.timers.retx:cancel())
   
   if self.state == SYN_SENT then
      inform_listeners(self, "connection_refused")
      self.state = CLOSED
      self.closed = true
      self.connected = false
      self.scheduler:enqueue(self.recvobj)
      return nil, "connection refused"
   elseif self.state == CLOSED or self.state == CLOSE_WAIT then
      self.state = CLOSED
      self.closed = true
      self.connected = false
      self.scheduler:enqueue(self.recvobj)
      return nil, "closed"
   end
   return true
end

function ClientSkt.send(self, data, i, j)
   if self.closed then
      return nil, "closed"
   end
   if self.out_shut then
      return nil, "socket output is shutdown"
   end
   if not self.connected then
      return nil, "connection reset"
   end
   local data = tostring(data)
   if i or j then
      data = data:sub(i, j)
   end
   local len = #data
   local total_bytes = 0
   local last_seq = 0xFF
   while total_bytes < len do
      while self.is_reset do
         self.scheduler:wait(self.reset_lock)
      end
      local write_bytes = math.min(self.cfg.max_seg_size
         - segments.RUDP_HEADER_LEN, len - total_bytes)
      last_seq = self.ctr:next_seqn()
      local pkt_data = data:sub(total_bytes + 1, total_bytes + write_bytes)
--[[
if (#self.unacked_sent_q == self.cfg.max_sendq_size) then
   while #self.unacked_sent_q > (self.cfg.max_sendq_size / 2) do
      self.scheduler:wait(self.unacked_send_lock)
   end
end
]]
--[[
      while CMP(last_seq, self.ctr:get_last()) > 0 do
         self.scheduler:wait(self.unacked_send_lock)
      end
]]
      local dat = segments.new("DAT", last_seq, self.ctr:get_last(), pkt_data)
      local ok, err = send_and_q_seg(self, dat)
      if not ok then
         return nil, err, total_bytes
      end
      total_bytes = total_bytes + write_bytes
   end
   return total_bytes
end

---
-- Resets the socket state and profile.
-- The socket will attempt to deliver all outstanding bytes to the remote
-- endpoint and then it will renegotiate the connection parameters specified
-- in the given socket profile. The transmissions of bytes resumes after the
-- renegotation finishes and the connection is synchronized again.
function ClientSkt.reset(self, config)
   if self.closed then
      return nil, "closed"
   end
   if not self.connected then
      return nil, "socket is not connected"
   end
   self.is_reset = true
   send_and_q_seg(segments.new("RST", self.ctr:next_seqn()))
   -- Wait to flush all outstanding segments (including last RST segment).
   while #self.unacked_sent_q ~= 0 do
      self.scheduler:wait(self.unacked_reset_lock)
   end
   inform_listeners(self, "connection_reset")
   -- Set new profile
   if config then
      self.cfg = config
   end
   -- Synchronize sequence numbers
   self.state = SYN_SENT
   send_and_q_seg(self, make_syn(self))
end

local function read_loop(self, check_buf, handle_data, ...)
   local ok = true
   local total = 0
   local out = {}
   local timeout_time = (self.timeout and self.timeout > 0) and self.scheduler:time() + self.timeout

   local ret, data, rest = check_buf(self, out, ...)
   if ret then
      self.buf = rest
      return data
   else
      total = #self.buf
      table.insert(out, self.buf)
      self.buf = ""
   end
    
   while true do
      while #self.inseq_recv_q == 0 do
         if self.closed then
            return nil, "closed", table.concat(out)
         end
         if self.in_shut then
            return nil, "eof", table.concat(out)
         end
         if not self.connected then
            return nil, "connection reset", table.concat(out)
         end
         local ok, err = self.scheduler:wait_until(self.recv_q_lock, timeout_time)
         if err == "timeout" then
            return nil, "timeout", table.concat(out)
         end
      end
      for i, s, remove in utils.ipairs_remove(self.inseq_recv_q) do
         remove()
         if s.type == "RST" then
            break
         elseif s.type == "FIN" then
            return nil, "eof", table.concat(out)
         elseif s.type == "DAT" then
            local data, rest = handle_data(self, s.data, total, ...)
            if rest then
               table.insert(out, data)
               self.buf = rest
               return table.concat(out)
            else
               table.insert(out, s.data)
               total = total + #s.data
            end
         end
      end
      
   end
end

local function break_upto(s, at, strip_nl)
   local data = s:sub(1, at)
   if strip_nl then data = data:match("(.-)[\r\n]+$") end
   return data, s:sub(at+1)
end

local read_bytes
do
   local function check_buf_bytes(self, out, wanted)
      if wanted <= #self.buf then
         return true, break_upto(self.buf, wanted)
      end
   end
   local function handle_data_bytes(self, data, total, wanted)
      if total + #data >= wanted then
         return break_upto(data, wanted - total)
      end
   end
   read_bytes = function(self, wanted)
      return read_loop(self, check_buf_bytes, handle_data_bytes, wanted)
   end
end

local read_buffer
do
   local function check_buf_buffer(self, out)
   end
   local function handle_data_buffer(self, data)
      local last = nil
      if #self.inseq_recv_q == 0 then
         last = ""
      end
      return data, last
   end
   read_buffer = function(self, wanted)
      return read_loop(self, check_buf_buffer, handle_data_buffer)
   end
end

local read_line
do
   local function check_buf_line(self, out, strip_nl)
      local nl = self.buf:find("\n")
      if nl then
         return true, break_upto(self.buf, nl, strip_nl)
      end
   end
   local function handle_data_line(self, data, total, strip_nl)
      local nl = data:find("\n")
      if nl then
         return break_upto(data, nl, strip_nl)
      end
   end
   read_line = function(self, strip_nl)
      return read_loop(self, check_buf_line, handle_data_line, strip_nl)
   end
end

local read_all
do
   local function keep_going()
   end
   read_all = function(self)
      return read_loop(self, keep_going, keep_going)
   end
end

function ClientSkt.receive(self, mode)
   local out, err, partial
   local tm = self.scheduler:time()
   if type(mode) == "number" then
      return read_bytes(self, mode)
   elseif mode == "*b" then
      return read_buffer(self)
   elseif mode == "*l" then
      return read_line(self, true)
   elseif mode == "*L" then
      return read_line(self, false)
   elseif mode == "*a" then
      local _, err, data = read_all(self)
      if err ~= "timeout" then
         return data
      else
         return nil, err, data
      end
   end
   return nil, "invalid mode"
end

function ClientSkt.flush(self)
   -- essa funcao custa muito (porque temos que esperar todos os pacotes serem reconhecidos ACK)
   while #self.unacked_sent_q ~= 0 do
      self.scheduler:wait(self.unacked_flush_lock)
   end
end

function ClientSkt.last_seqn(self)
   return self.ctr:get_last()
end

---
-- Lower-level close function, which cleans up and closes the socket.
local function close_impl(self)
   assert(self.timers.nilseg:cancel())
   assert(self.timers.keepalive:cancel())
   self.state = CLOSE_WAIT
   local closing = coroutine.create(function()
      self.scheduler:set_thread_name("closing")
      self.closed = true
      assert(self.timers.keepalive:cancel())
      assert(self.timers.nilseg:cancel())
      assert(self.timers.uidseg:cancel())
      self.scheduler:sleep(3)
      assert(self.timers.retx:cancel())
      assert(self.timers.cack:cancel())
      if self.own_skt then
         self.skt:close()
      end
      self.scheduler:enqueue(self.recvobj)
      inform_listeners(self, "connection_closed")
   end)
   local ok, err = coroutine.resume(closing)
   if not ok then error(err, 2) end
end

local function notify_unacked_sent_q(self)
   self.scheduler:notify(self.unacked_sent_q)
   self.scheduler:notify(self.unacked_flush_lock)
   self.scheduler:notify(self.unacked_reset_lock)
   self.scheduler:notify(self.unacked_send_lock)
end

function ClientSkt.close(self)
   if self.closed then
      return
   end
   -- TODO remove shutdown hook
   local state = self.state
   if state == SYN_SENT then
      self.scheduler:notify(self.connection_lock)
   elseif state == CLOSE_WAIT or state == SYN_RCVD or state == ESTABLISHED then
      send_seg(self, segments.new("FIN", self.ctr:next_seqn()))
      close_impl(self)
      return
   elseif state == CLOSED then
      assert(self.timers.retx:cancel())
      assert(self.timers.cack:cancel())
      assert(self.timers.keepalive:cancel())
      assert(self.timers.nilseg:cancel())
   end
   self.state = CLOSED
   self.closed = true
   notify_unacked_sent_q(self)
   self.scheduler:notify(self.recv_q_lock)
end

function ClientSkt.set_keepalive(self, on)
   if self.keepalive == on then
      return
   end
   self.keepalive = on
   if self.connected then
      assert(self.timers.keepalive:cancel())
      if on then
         local time = self.cfg.nilseg_to * 10
         assert(self.timers.keepalive:schedule(time, time))
      end
   end
end

function ClientSkt.set_uuid(self, uuid)
   self.uuid = utils.binary_uuid(uuid)
end

function ClientSkt.shutdown_input(self)
   if self.closed then
      return nil, "closed"
   end
   if not self.connected then
      return nil, "socket is not connected"
   end
   if self.in_shut then
      return nil, "socket input is already shutdown"
   end
   self.in_shut = true
   self.scheduler:notify(self.recv_q_lock)
end

function ClientSkt.shutdown_output(self)
   if self.closed then
      return nil, "closed"
   end
   if not self.connected then
      return nil, "socket is not connected"
   end
   if self.out_shut then
      return nil, "socket output is already shutdown"
   end
   self.out_shut = true
   notify_unacked_sent_q(self)
end

function ClientSkt.getsockname(self)
   return self.skt:getsockname()
end

---
-- Sends an ACK segment if there is a received segment to be acknowledged.
local function send_single_ack(self)
   if self.ctr:getreset_cack() == 0 then
      return
   end
   local last = self.ctr:get_last()
   send_seg(self, segments.new("ACK", counter.next(last), last))
end

---
-- Sends an EAK segment if there is at least one out-of-sequence received
-- segment.
local function send_extended_ack(self)
   if not next(self.outseq_recv_q) then
      return
   end
   self.ctr:getreset_cack()
   self.ctr:getreset_outseq()
   -- Compose list of out-of-sequence sequence numbers
   local acks = {}
   for i, seg in ipairs(self.outseq_recv_q) do
      acks[i] = seg.seqn
   end
   local last = self.ctr:get_last()
   send_seg(self, segments.new("EAK", counter.next(last), last, acks))
end

---
-- Acknowledges the next segment to be acknowledged. If there are any
-- out-of-sequence segments in the receiver queue, it sends an EAK segment.
local function send_ack(self)
   if next(self.outseq_recv_q) then
      send_extended_ack(self)
   else
      send_single_ack(self)
   end
end

---
-- Puts the connection in an "opened" state
local function connection_opened(self)
   if self.connected then
      assert(self.timers.nilseg:cancel())
      assert(self.timers.uidseg:cancel())
      if self.keepalive then
         assert(self.timers.keepalive:cancel())
      end
      self.is_reset = false
      self.scheduler:notify(self.reset_lock)
   else
      self.connected = true
      self.state = ESTABLISHED
      self.scheduler:notify(self.connection_lock)
      inform_listeners(self, "connection_opened")
   end
   local timeout = self.cfg.nilseg_to
   assert(self.timers.nilseg:schedule(timeout, timeout))
   assert(self.timers.uidseg:schedule(timeout, timeout))
   if self.keepalive then
      -- may reschedule
      self.timers.keepalive:schedule(timeout * 10, timeout * 10)
   end
end

---
-- Handles a received SYN segment.
-- 
-- When a client initiates a connection it sends a SYN segment which
-- contains the negotiable parameters defined by the Upper Layer Protocol
-- via the API. The server can accept these parameters by echoing them back
-- in its SYN with ACK response or propose different parameters in its SYN
-- with ACK response. The client can then choose to accept the parameters
-- sent by the server by sending an ACK to establish the connection or it
-- can refuse the connection by sending a FIN.
local function handle_syn(self, seg)
   if self.state == CLOSED then
      self.ctr:set_last(seg.seqn)
      self.state = SYN_RCVD
      self.cfg = config.new(self.cfg.max_sendq_size, self.cfg.max_recvq_size,
         seg.max_seg_size, seg.max_ostand, seg.max_retx, seg.max_cack,
         seg.max_outseq, seg.max_autorst, seg.nilseg_to, seg.retx_to,
         seg.cack_to)
      local seqn = self.ctr:set_seqn(random_seqn())
      local syn = segments.new("SYN", seqn,
         self.cfg.max_ostand, self.cfg.max_seg_size, self.cfg.retx_to,
         self.cfg.cack_to, self.cfg.nilseg_to,
         self.cfg.max_retx, self.cfg.max_cack, self.cfg.max_outseq,
         self.cfg.max_autorst)
      syn:set_ack(seg.seqn)
      send_and_q_seg(self, syn)
   elseif self.state == SYN_SENT then
      self.ctr:set_last(seg.seqn)
      self.state = ESTABLISHED
      -- Here the client accepts or rejects the parameters sent by the
      -- server. For now we will accept them.
      send_ack(self)
      connection_opened(self)
   else
      -- segment received in other state. drop.
   end
end

---
-- Checks for in-sequence segments in the out-of-sequence queue that can be
-- moved to the in-sequence queue.
local function check_recv_qs(self)
   for i, seg, remove in utils.ipairs_remove(self.outseq_recv_q) do
      if CMP(seg.seqn, counter.next(self.ctr:get_last())) == 0 then
         self.ctr:set_last(seg.seqn)
         if seg.type == "DAT" or seg.type == "RST" or seg.type == "FIN" then
            table.insert(self.inseq_recv_q, seg)
         end
         remove()
      end
   end
   local ok, err = self.scheduler:notify(self.recv_q_lock)
end

---
-- Handles a received RST, FIN, or DAT segment.
local function handle_seg(self, seg)
   if seg.type == "RST" then
      -- When a RST segment is received, the sender must stop sending new
      -- packets, but most continue to attempt delivery of packets already
      -- accepted from the application.
      self.is_reset = true
      inform_listeners(self, "connection_reset")
   elseif seg.type == "FIN" then
      -- When a FIN segment is received, no more packets are expected to
      -- arrive after this segment.
      if self.state == SYN_SENT then
         self.scheduler:notify(self.connection_lock)
      elseif self.state ~= CLOSED then
         self.state = CLOSE_WAIT
         self:close()
      end
   end
   local inseq = false
   local last = self.ctr:get_last()
   if CMP(seg.seqn, last) <= 0 then
      -- drop packet: duplicate
   elseif CMP(seg.seqn, counter.next(last)) == 0 then
      inseq = true
      if #self.inseq_recv_q == 0 or #self.inseq_recv_q + #self.outseq_recv_q < self.recv_q_size then
         -- insert in-sequence segment
         self.ctr:set_last(seg.seqn)
         if seg.type == "DAT" or seg.type == "RST" or seg.type == "FIN" then
            table.insert(self.inseq_recv_q, seg)
         end
         if seg.type == "DAT" then
            inform_listeners(self, "packet_received_in_order")
         end
         check_recv_qs(self)
      else
         check_recv_qs(self)
         -- drop packet: queue is full
      end
   elseif #self.inseq_recv_q + #self.outseq_recv_q < self.recv_q_size then
      -- insert out-of-sequence segment, in order
      local added = false
      for i = 1, #self.outseq_recv_q do
         local s = self.outseq_recv_q[i]
         local cmp = CMP(seg.seqn, s.seqn)
         if cmp == 0 then
            -- ignore duplicate packet
            added = true
            break
         elseif cmp < 0 then
            table.insert(self.outseq_recv_q, i, seg)
            added = true
            break
         end
      end
      if not added then
         table.insert(self.outseq_recv_q, seg)
      end
      self.ctr:inc_outseq()
      if seg.type == "DAT" then
         inform_listeners(self, "packet_received_out_of_order")
      end
   end
   local should_ack = {RST=1,NUL=1,FIN=1,UID=1}
   if inseq and should_ack[seg.type] then
      send_ack(self)
   else
      local ctr_outseq = self.ctr:get_outseq()
      local max_outseq = self.cfg.max_outseq
      if ctr_outseq > 0 and (max_outseq == 0 or ctr_outseq > max_outseq) then
         send_extended_ack(self)
      else
         local ctr_cack = self.ctr:get_cack()
         local max_cack = self.cfg.max_cack
         if ctr_cack > 0 and (max_cack == 0 or ctr_cack > max_cack) then
            send_single_ack(self)
         else
            if self.timers.cack.idle then
               assert(self.timers.cack:schedule(self.cfg.cack_to))
            end
         end
      end
   end
end

---
-- Checks the ACK flag and number of a segment.
local function check_and_get_ack(self, seg)
   local ackn = seg.ackn
   if ackn == 0xFF then
      return
   end
   self.ctr:getreset_ostand()
   if self.state == SYN_RCVD then
      self.state = ESTABLISHED
      connection_opened(self)
   end
   local size = #self.unacked_sent_q
   for i = size, 1, -1 do
      if CMP(self.unacked_sent_q[i].seqn, ackn) <= 0 then
         table.remove(self.unacked_sent_q, i)
      end
   end
   if #self.unacked_sent_q == 0 then
      assert(self.timers.retx:cancel())
      inform_listeners(self, "ack_received", ackn)
   end
   notify_unacked_sent_q(self)
end

---
-- Puts the connection in a closed state
local function connection_failure(self)
   if self.closed then
      return
   end
   local state = self.state
   if state == SYN_SENT then
      self.scheduler:notify(self.connection_lock)
   elseif state == CLOSE_WAIT or state == SYN_RCVD or state == ESTABLISHED then
      self.connected = false
      notify_unacked_sent_q(self)
      self.scheduler:notify(self.recv_q_lock)
      close_impl(self)
   end
   self.state = CLOSED
   self.closed = true
   inform_listeners(self, "connection_failure")
end

---
-- Sends a segment and increments its retransmission counter.
local function retransmit_seg(self, seg)
   if self.cfg.max_retx > 0 then
      seg.nretx = seg.nretx + 1
   end
   if self.cfg.max_retx ~= 0 and seg.nretx > self.cfg.max_retx then
      connection_failure(self)
   end
   send_seg(self, seg)
   inform_listeners(self, "packet_retransmitted")
end

---
-- Handles a received EAK segment.
--
-- When a EAK segment is received, the segments specified in the message are
-- removed from the unacknowledged sent queue. The segments to be
-- retransmitted are determined by examining the Ack Number and the last out
-- of sequence ack number in the EAK segment. All segments between but not
-- including these two sequence numbers that are on the unacknowledged sent
-- queue are retransmitted.
local function handle_eak(self, seg)
   local acks = seg.acks
   local last_inseq = seg.ackn
   local last_outseq = acks[#acks]
   -- Remove acknowledged segments from sent queue
   for _, s, remove in utils.ipairs_remove(self.unacked_sent_q) do
      if CMP(s.seqn, last_inseq) <= 0 then
         remove()
      else
         for _, ack in ipairs(acks) do
            if CMP(s.seqn, ack) == 0 then
               remove()
               break
            end
         end
      end
   end
   -- Retransmit segments
   for _, s in ipairs(self.unacked_sent_q) do
      if CMP(last_inseq, s.seqn) < 0 and CMP(last_outseq, s.seqn) > 0 then
         retransmit_seg(self, s)
      end
   end
   notify_unacked_sent_q(self)
end

receiver_fn = function(self)
   self.scheduler:set_thread_name("client receiver_fn")
   local inc_cack_segs = {DAT=1,NUL=1,RST=1,FIN=1,SYN=1}
   while true do
      local ok, seg, host, port
      local brk = false
      repeat
         if seg then
            tests.log_packet(seg, "LOST")
         end
         ok, seg, host, port = self.scheduler:wait(self.recvobj)
         if seg and type(seg) == "string" then
            seg = segments.parse(seg)
         end
         if not seg then
            brk = true
            break
         end
      until ((not tests.lose_packets) or math.random(1,100) > tests.packet_loss_percent)
      if brk then break end
      tests.log_packet(seg, "recv")
      if inc_cack_segs[seg.type] then
         self.ctr:inc_cack()
      end
      if self.keepalive then
         assert(self.timers.keepalive:reset())
      end
      if seg.type == "SYN" then
         handle_syn(self, seg)
      elseif seg.type == "EAK" then
         handle_eak(self, seg)
      elseif seg.type == "ACK" or seg.type == "UID" then
         -- do nothing
      else
         handle_seg(self, seg)
      end
      check_and_get_ack(self, seg)
   end
   self.receiver_coro = nil
end

ClientSkt.timers = {
   
   ---
   -- This timer is started when the connection is opened and is reset every
   -- time a data segment is sent. If the client's null segment timer expires,
   -- the client sends a null segment to the server.
   nilseg = function(tmr, self)
      if #self.unacked_sent_q == 0 then
         local seqn = self.ctr:next_seqn()
         send_and_q_seg(self, segments.new("NUL", seqn))
      end
   end,
   
   uidseg = function(tmr, self)
      if self.uuid then
         send_seg(self, segments.new("UID", 0xFF, self.uuid))
      end
   end,
   
   ---
   -- This timer is re-started every time a data, null, or reset segment is
   -- sent and there is not a segment currently being timed. If an
   -- acknowledgment for this data segment is not received by the time the
   -- timer expires, all segments that have been sent but not acknowledged are
   -- retransmitted. The Retransmission timer is re-started when the timed
   -- segment is received, if there is still one or more packets that have been
   -- sent but not acknowledged.
   retx = function(tmr, self)
      for _, seg in ipairs(self.unacked_sent_q) do
         retransmit_seg(self, seg)
      end
   end,

   ---
   -- When this timer expires, if there are segments on the out-of-sequence
   -- queue, an extended acknowledgment is sent. Otherwise, if there are any
   -- segments currently unacknowledged, a stand-alone acknowledgment is sent.
   -- The cumulative acknowledge timer is restarted whenever an acknowledgment
   -- is sent in a data, null, or reset segment, provided that there are no
   -- segments currently on the out-of-sequence queue. If there are segments on
   -- the out-of-sequence queue, the timer is not restarted, so that another
   -- extended acknowledgment will be sent when it expires again.
   cack = function(tmr, self)
      send_ack(self)
   end,

   ---   
   -- When this timer expires, the connection is considered broken.
   keepalive = function(tmr, self)
      connection_failure(self)
   end,
   
}

ClientSkt.mt = {
   __gc = function(self)
      self:close()
   end,
}

--- Create a new client socket.
-- @param address string: the IP address as a string.
-- @param port number: the port number
-- @return a socket object.
function client.new(scheduler, address, port, server, config)
   assert(scheduler, "scheduler not given")
   if not config then config = default_config end
   local self = {
      -- private data:
      buf = "",
      own_skt = (server ~= nil),
      scheduler = scheduler,
      receiver_coro = nil,
      cfg = config,
      ctr = counter.new(),
      unacked_sent_q = {},
      outseq_recv_q = {},
      inseq_recv_q = {},
      send_q_size = 32,
      recv_q_size = 32,
      send_buf_size = (config.max_seg_size - segments.RUDP_HEADER_LEN) * 32,
      recv_buf_size = (config.max_seg_size - segments.RUDP_HEADER_LEN) * 32,
      connection_lock = {},
      unacked_flush_lock = {},
      unacked_reset_lock = {},
      unacked_send_lock = {},
      reset_lock = {},
      send_and_q_mutex = mutex.new(scheduler),
      -- TODO close_lock = {},
      recv_q_lock = {},
      listeners = {},
      is_reset = false,
      state = CLOSED,

      -- gettable fields:
      host = address,
      port = port,
      timeout = nil, -- seconds; nil means no timeout
      closed = false,
      connected = false,
      keepalive = true,
      in_shut = false,
      out_shut = false,

      -- gettable/settable fields:
      uuid = nil,
      
      -- methods:
      connect = ClientSkt.connect,      
      reset = ClientSkt.reset,
      send = ClientSkt.send,
      receive = ClientSkt.receive,
      flush = ClientSkt.flush,
      close = ClientSkt.close,
      getsockname = ClientSkt.getsockname,
      set_keepalive = ClientSkt.set_keepalive,
      set_uuid = ClientSkt.set_uuid,
      shutdown_input = ClientSkt.shutdown_input,
      shutdown_output = ClientSkt.shutdown_output,
      add_listener = ClientSkt.add_listener,
      remove_listener = ClientSkt.remove_listener,
      last_seqn = ClientSkt.last_seqn,
   }
   self.timers = {}
   for name, timer_fn in pairs(ClientSkt.timers) do
      self.timers[name] = timer.new(scheduler, name, timer_fn, self)
   end
   -- TODO shutdown hook?
   local ok, err
   if server then
      self.skt = server.skt
      self.recvobj = self
      self.receiver_coro = coroutine.create(receiver_fn)
      coroutine.resume(self.receiver_coro, self)
   else
      self.skt = socket.udp()
      self.recvobj = self.skt
      if address and port then
         ok, err = self:connect(address, port, 0)
         if not ok then
            return nil, err
         end
      end
   end
   setmetatable(self, ClientSkt.mt)
   
   return self, err
end

return client
