
local counter = {}

local MAX_SEQUENCE_NUMBER = 254

--- Compares two sequence numbers
-- This algorithm is useful for sliding window comparisons that wrap around.
-- @return 0, 1 or -1 if the first sequence number is equal, greater or less
-- than the second sequence number. (see RFC 1982).
function counter.compare_seqn(seqn, aseqn)
   if seqn == aseqn then
      return 0
   elseif (((seqn < aseqn) and ((aseqn - seqn) > MAX_SEQUENCE_NUMBER / 2)) or ((seqn > aseqn) and ((seqn - aseqn) < MAX_SEQUENCE_NUMBER / 2))) then
      return 1
   end
   return -1
end

function counter.next(n)
   return (n+1) % (MAX_SEQUENCE_NUMBER+1)
end

function counter.new()
   return {
      -- Segment sequence number
      seqn = 0,
      -- Last in-sequence received segment
      last = 0,
      -- Cumulative acknowledge counter:
      -- The receiver maintains a counter of unacknowledged segments received
      -- without an acknowledgment being sent to the transmitter. The maximum
      -- value of this counter is configurable. If this counter's maximum is
      -- exceeded, the receiver sends either a stand-alone acknowledgment, or
      -- an extended acknowledgment if there are currently any out-of-sequence
      -- segments. The recommended value for the cumulative acknowledge
      -- counter is 3.
      cack = 0,
      -- Out-of-sequence acknowledgments counter:
      -- The receiver maintains a counter of the number of segments that have
      -- arrived out-of-sequence. Each time this counter exceeds its
      -- configurable maximum, an extended acknowledgment segment containing
      -- the sequence numbers of all current out-of-sequence segments that
      -- have been received is sent to the transmitter. The counter is then
      -- reset to zero. The recommended value for the out-of-sequence
      -- acknowledgments counter is 3.
      outseq = 0,
      -- Outstanding segments counter:
      -- The transmitter maintains a counter of the number of segments that
      -- have been sent without getting an acknowledgment. This is used by the
      -- receiver as a mean of flow control.
      ostand = 0,

      next_seqn = function(self) self.seqn = counter.next(self.seqn); return self.seqn; end,
      set_seqn = function(self, n) self.seqn = n; return n; end,
      get_seqn = function(self) return self.seqn; end,

      set_last = function(self, n) self.last = n; return n; end,
      get_last = function(self) return self.last; end,
      
      inc_cack = function(self) self.cack = self.cack + 1 end,
      get_cack = function(self) return self.cack end,
      getreset_cack = function(self) local n = self.cack; self.cack = 0; return n; end,

      inc_outseq = function(self) self.outseq = self.outseq + 1 end,
      get_outseq = function(self) return self.outseq end,
      getreset_outseq = function(self) local n = self.outseq; self.outseq = 0; return n; end,

      inc_ostand = function(self) self.ostand = self.ostand + 1 end,
      get_ostand = function(self) return self.ostand end,
      getreset_ostand = function(self) local n = self.ostand; self.ostand = 0; return n; end,
      
      reset = function(self)
         self.outseq = 0
         self.ostand = 0
         self.cack = 0
      end,
   }
end

counter.MAX_SEQUENCE_NUMBER = MAX_SEQUENCE_NUMBER

return counter
