
local utils = {}

--- Get a binary representation of a string containing a UUID.
-- @param uuid If it is a hex representation of a UUID, convert it to
-- binary; if it is binary already, just return it.
-- @return a 16-byte binary string with the given UUID.
function utils.binary_uuid(uuid)
   if #uuid == 16 then
      return uuid
   else
      uuid = uuid:gsub("[^0-9a-fA-F]", "")
      local bytes = {}
      for hex in uuid:gmatch("..") do
         bytes[#bytes+1] = tonumber(hex, 16)
      end
      return string.char(table.unpack(bytes))
   end
end

--- Get a text representation of a binary UUID.
-- @param uuid If it is a binary representation of a UUID, convert it to
-- text; if it is text already, just return it.
-- @return a 32-char hex representation of the given UUID.
function utils.text_uuid(uuid)
   if #uuid == 16 then
      local out = {}
      for c in uuid:gmatch(".") do
         table.insert(out, ("%02x"):format(c:byte()))
      end
      return table.concat(out)
   else
      return uuid
   end
end

function utils.ipairs_remove(tbl)
   local i = 0
   local remove = function()
      table.remove(tbl, i)
      i = i - 1
   end
   return function()
      i = i + 1
      if tbl[i] then
         return i, tbl[i], remove
      end
      return nil
   end
end

function utils.add_to_listeners_tbl(listeners, listener, arg)
   for _, l in ipairs(listeners) do
      if l.listener == listener then
         return
      end
   end
   table.insert(listeners, { listener = listener, arg = arg })
end

function utils.remove_from_listeners_tbl(listeners, listener)
   for i, l in ipairs(listeners) do
      if l.listener == listener then
         table.remove(listeners, i)
         return
      end
   end
end

function utils.inform_listeners(listeners, self, event, ...)
   for i, entry in ipairs(listeners) do
      local handler = entry.listener[event]
      if handler then
         local handler_coro = coroutine.create(handler)
         self.scheduler:set_thread_name(event.." handler "..i, handler_coro)
         local ok, err = coroutine.resume(handler_coro, entry.arg, self, ...)
         if not ok then
            print(err)
         end
      end
   end
end

return utils

